<?php
require_once __DIR__ . '/../../src/config.php';
require_once __DIR__ . '/../../src/layout.php';

if (session_status() !== PHP_SESSION_ACTIVE) {
  session_start();
}

if (!isset($_SESSION['user_id'])) {
  header('Location: ' . app_href('login.php'));
  exit;
}

$pdo = db();
$stmt = $pdo->prepare('SELECT id, name, email, user_type FROM users WHERE id = ?');
$stmt->execute([$_SESSION['user_id']]);
$user = $stmt->fetch();
if (!$user) {
  header('Location: ' . app_href('login.php'));
  exit;
}

$isStudent = ($user['user_type'] === 'student');
$title = $isStudent ? 'Welcome back' : 'Dashboard';
$subtitle = $isStudent ? 'Your classes, teachers, and papers at a glance.' : 'Overview';

render_header($title, [], $user);
?>

<!-- Hero Welcome Section -->
<section style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border-radius: 12px; padding: 3rem; color: white; margin-bottom: 3rem;">
  <div style="position: absolute; top: 0; right: 0; font-size: 8rem; opacity: 0.1; position: relative;">
    <i class="bi bi-book-half"></i>
  </div>
  <div class="position-relative">
    <h1 class="mb-2" style="font-size: 2.5rem; font-weight: 700;">
      <i class="bi bi-hand-thumbs-up-fill me-2"></i>Welcome back, <?= htmlspecialchars(explode(' ', $user['name'])[0]) ?>!
    </h1>
    <p style="opacity: 0.95; font-size: 1.1rem; margin: 0;">Your classes, teachers, and papers at a glance</p>
  </div>
</section>

<div style="display: grid; grid-template-columns: 1fr 1fr; gap: 2rem; margin-bottom: 3rem;">
  <!-- Left Column: Teachers & Papers -->
  <div>
    <!-- Your Teachers Section -->
    <section style="background: white; border-radius: 12px; padding: 0; margin-bottom: 2rem; box-shadow: 0 2px 8px rgba(0,0,0,0.08); overflow: hidden;">
      <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 1.5rem; color: white; border-bottom: 4px solid #667eea;">
        <h2 class="h5 mb-0 d-flex align-items-center gap-2" style="font-weight: 700;">
          <i class="bi bi-person-video3" style="font-size: 1.3rem;"></i>Your Teachers
        </h2>
      </div>
      <div style="padding: 1.5rem;">
        <?php
        try {
          $q = $pdo->prepare('SELECT t.id, u.name, u.email
                               FROM teacher_student ts
                               JOIN users u ON u.id = ts.teacher_id
                               JOIN users t ON t.id = ts.student_id
                               WHERE ts.student_id = ? AND u.user_type = "teacher"');
          $q->execute([$user['id']]);
          $teachers = $q->fetchAll();
        } catch (Throwable $e) {
          $teachers = [];
        }
        if (!$teachers): ?>
          <div class="alert alert-warning d-flex align-items-center gap-2 mb-0" style="border-left: 4px solid #ffc107; border-radius: 8px; background-color: #fff8e1;">
            <i class="bi bi-exclamation-triangle-fill" style="font-size: 1.2rem;"></i>
            <span>No teachers linked yet. Please contact support or your school to be assigned.</span>
          </div>
        <?php else: ?>
          <div style="display: flex; flex-direction: column; gap: 1rem;">
            <?php foreach ($teachers as $t): ?>
              <div style="padding: 1rem; background: #f9f9f9; border-radius: 10px; border-left: 4px solid #667eea; transition: all 0.2s;" 
                   onmouseover="this.style.backgroundColor='#f0f0f0'; this.style.boxShadow='0 4px 12px rgba(0,0,0,0.1)';" 
                   onmouseout="this.style.backgroundColor='#f9f9f9'; this.style.boxShadow='none';">
                <div class="d-flex justify-content-between align-items-center gap-2">
                  <div style="flex: 1;">
                    <p style="margin: 0 0 0.3rem 0; font-weight: 700; color: #333; font-size: 1rem;">
                      <?= htmlspecialchars($t['name']) ?>
                    </p>
                    <p style="margin: 0; color: #999; font-size: 0.9rem;">
                      <i class="bi bi-envelope me-1"></i><?= htmlspecialchars($t['email']) ?>
                    </p>
                  </div>
                  <a href="mailto:<?= htmlspecialchars($t['email']) ?>" class="btn btn-sm" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none; font-weight: 600; white-space: nowrap;">
                    <i class="bi bi-envelope-fill"></i>
                  </a>
                </div>
              </div>
            <?php endforeach; ?>
          </div>
        <?php endif; ?>
      </div>
    </section>

    <!-- Papers Section -->
    <section style="background: white; border-radius: 12px; padding: 0; box-shadow: 0 2px 8px rgba(0,0,0,0.08); overflow: hidden;">
      <div style="background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%); padding: 1.5rem; color: white; border-bottom: 4px solid #4facfe;">
        <h2 class="h5 mb-0 d-flex align-items-center gap-2" style="font-weight: 700;">
          <i class="bi bi-file-earmark-text" style="font-size: 1.3rem;"></i>Your Papers
        </h2>
      </div>
      <div style="padding: 1.5rem;">
        <?php
        // Fetch published papers for mapped teachers
        $pSql = "SELECT p.id, p.title, p.time_limit_seconds, u.name AS teacher_name
                 FROM papers p
                 JOIN users u ON u.id = p.teacher_id
                 JOIN teacher_student ts ON ts.teacher_id = p.teacher_id AND ts.student_id = ?
                 WHERE p.is_published = 1";
        $pStmt = $pdo->prepare($pSql);
        $pStmt->execute([$user['id']]);
        $allPapers = $pStmt->fetchAll();

        // Fetch attempts for this student
        $aStmt = $pdo->prepare('SELECT id, paper_id, score, submitted_at, started_at FROM attempts WHERE student_id = ?');
        $aStmt->execute([$user['id']]);
        $attemptsByPaper = [];
        foreach ($aStmt->fetchAll() as $a) { $attemptsByPaper[$a['paper_id']] = $a; }

        $completed = [];
        $inprogress = [];
        $unstarted = [];
        foreach ($allPapers as $p) {
          if (isset($attemptsByPaper[$p['id']]) && $attemptsByPaper[$p['id']]['submitted_at']) {
            $completed[] = [
              'paper' => $p,
              'attempt' => $attemptsByPaper[$p['id']]
            ];
          } elseif (isset($attemptsByPaper[$p['id']]) && !$attemptsByPaper[$p['id']]['submitted_at']) {
            $inprogress[] = [
              'paper' => $p,
              'attempt' => $attemptsByPaper[$p['id']]
            ];
          } else {
            $unstarted[] = $p;
          }
        }
        ?>

        <!-- Completed Papers -->
        <div class="mb-4">
          <h6 style="font-weight: 700; color: #333; margin-bottom: 1rem; display: flex; align-items: center; gap: 0.5rem;">
            <i class="bi bi-check-circle-fill" style="color: #00d084; font-size: 1.2rem;"></i> Completed (<?= count($completed) ?>)
          </h6>
          <?php if (empty($completed)): ?>
            <p style="color: #999; text-align: center; padding: 1rem;">No completed papers yet</p>
          <?php else: ?>
            <div style="display: flex; flex-direction: column; gap: 1rem;">
              <?php foreach ($completed as $c): $p=$c['paper']; $a=$c['attempt']; ?>
                <div style="padding: 1rem; background: #f0fdf4; border: 2px solid #00d084; border-radius: 10px; transition: all 0.2s;"
                     onmouseover="this.style.boxShadow='0 4px 12px rgba(0,212,132,0.2)';"
                     onmouseout="this.style.boxShadow='none';">
                  <div class="d-flex justify-content-between align-items-start gap-2">
                    <div style="flex: 1;">
                      <p style="margin: 0 0 0.3rem 0; font-weight: 700; color: #333;">
                        <?= htmlspecialchars($p['title']) ?>
                      </p>
                      <p style="margin: 0 0 0.5rem 0; color: #666; font-size: 0.9rem;">
                        <i class="bi bi-person me-1" style="color: #4facfe;"></i><?= htmlspecialchars($p['teacher_name']) ?>
                      </p>
                      <p style="margin: 0; color: #999; font-size: 0.85rem;">
                        <i class="bi bi-calendar me-1"></i>Submitted <?= htmlspecialchars(date('M d, Y H:i', strtotime($a['submitted_at']))) ?>
                      </p>
                    </div>
                    <div style="text-align: right;">
                      <div style="background: linear-gradient(135deg, #00d084 0%, #13c084 100%); color: white; padding: 0.5rem 1rem; border-radius: 6px; font-weight: 700; font-size: 1.2rem; margin-bottom: 0.5rem;">
                        <?= (int)$a['score'] ?>
                      </div>
                      <a class="btn btn-sm btn-outline-primary" style="font-weight: 600;" href="<?= htmlspecialchars(app_href('student/result.php?attempt_id=' . $a['id'])) ?>">
                        <i class="bi bi-eye-fill me-1"></i>View
                      </a>
                    </div>
                  </div>
                </div>
              <?php endforeach; ?>
            </div>
          <?php endif; ?>
        </div>

        <!-- Unstarted Papers -->
        <div class="mb-4">
          <h6 style="font-weight: 700; color: #333; margin-bottom: 1rem; display: flex; align-items: center; gap: 0.5rem;">
            <i class="bi bi-clock-history" style="color: #ffc107; font-size: 1.2rem;"></i> Unstarted (<?= count($unstarted) ?>)
          </h6>
          <?php if (empty($unstarted)): ?>
            <p style="color: #999; text-align: center; padding: 1rem;">All papers started!</p>
          <?php else: ?>
            <div style="display: flex; flex-direction: column; gap: 1rem;">
              <?php foreach ($unstarted as $p): ?>
                <div style="padding: 1rem; background: #fffbf0; border: 2px solid #ffc107; border-radius: 10px; transition: all 0.2s;"
                     onmouseover="this.style.boxShadow='0 4px 12px rgba(255,193,7,0.2)';"
                     onmouseout="this.style.boxShadow='none';">
                  <div class="d-flex justify-content-between align-items-start gap-2">
                    <div style="flex: 1;">
                      <p style="margin: 0 0 0.3rem 0; font-weight: 700; color: #333;">
                        <?= htmlspecialchars($p['title']) ?>
                      </p>
                      <p style="margin: 0; color: #666; font-size: 0.9rem;">
                        <i class="bi bi-person me-1" style="color: #4facfe;"></i><?= htmlspecialchars($p['teacher_name']) ?> · 
                        <i class="bi bi-hourglass-split me-1" style="color: #ffc107;"></i><?= intval($p['time_limit_seconds'] / 60) ?> min
                      </p>
                    </div>
                    <a class="btn btn-sm" style="background: linear-gradient(135deg, #ffc107 0%, #ff9800 100%); color: white; border: none; font-weight: 600; white-space: nowrap;" href="<?= htmlspecialchars(app_href('student/attempt.php?paper_id=' . $p['id'])) ?>">
                      <i class="bi bi-play-fill"></i> Start
                    </a>
                  </div>
                </div>
              <?php endforeach; ?>
            </div>
          <?php endif; ?>
        </div>

        <!-- In Progress Papers -->
        <div>
          <h6 style="font-weight: 700; color: #333; margin-bottom: 1rem; display: flex; align-items: center; gap: 0.5rem;">
            <i class="bi bi-play-circle-fill" style="color: #4facfe; font-size: 1.2rem;"></i> In Progress (<?= count($inprogress) ?>)
          </h6>
          <?php if (empty($inprogress)): ?>
            <p style="color: #999; text-align: center; padding: 1rem;">No active attempts</p>
          <?php else: ?>
            <div style="display: flex; flex-direction: column; gap: 1rem;">
              <?php foreach ($inprogress as $c): $p=$c['paper']; $a=$c['attempt']; 
                $started = strtotime($a['started_at']);
                $deadline = $started + (int)$p['time_limit_seconds'];
                $now = time();
                $remain = max(0, $deadline - $now);
                $rm = (int)floor($remain/60); $rs = $remain % 60;
              ?>
                <div style="padding: 1rem; background: <?= $remain > 300 ? '#e7f3ff' : '#ffe7e7' ?>; border: 2px solid <?= $remain > 300 ? '#4facfe' : '#dc3545' ?>; border-radius: 10px; transition: all 0.2s;" 
                     data-attempt-id="<?= (int)$a['id'] ?>"
                     onmouseover="this.style.boxShadow='0 4px 12px rgba(0,0,0,0.1)';"
                     onmouseout="this.style.boxShadow='none';">
                  <div class="d-flex justify-content-between align-items-start gap-2">
                    <div style="flex: 1;">
                      <p style="margin: 0 0 0.3rem 0; font-weight: 700; color: #333;">
                        <?= htmlspecialchars($p['title']) ?>
                      </p>
                      <p style="margin: 0 0 0.5rem 0; color: #666; font-size: 0.9rem;">
                        <i class="bi bi-person me-1" style="color: #4facfe;"></i><?= htmlspecialchars($p['teacher_name']) ?>
                      </p>
                      <p style="margin: 0; color: <?= $remain > 300 ? '#0066cc' : '#dc3545' ?>; font-weight: 600; font-size: 0.9rem;">
                        <i class="bi bi-hourglass-split me-1"></i>
                        <span class="remain" data-seconds="<?= $remain ?>"><?= $rm ?>m <?= $rs ?>s</span> remaining
                      </p>
                    </div>
                    <a class="btn btn-sm resume-btn" data-paper-id="<?= (int)$p['id'] ?>" style="background: <?= $remain > 300 ? 'linear-gradient(135deg, #4facfe 0%, #00f2fe 100%)' : 'linear-gradient(135deg, #dc3545 0%, #c82333 100%)' ?>; color: white; border: none; font-weight: 600; white-space: nowrap;" href="<?= htmlspecialchars(app_href('student/attempt.php?paper_id=' . $p['id'])) ?>">
                      <i class="bi bi-play-fill"></i> Resume
                    </a>
                  </div>
                </div>
              <?php endforeach; ?>
            </div>
          <?php endif; ?>
        </div>
      </div>
    </section>
  </div>

  <!-- Right Column: Profile & Stats -->
  <div>
    <!-- Profile Card -->
    <section style="background: white; border-radius: 12px; padding: 0; margin-bottom: 2rem; box-shadow: 0 2px 8px rgba(0,0,0,0.08); overflow: hidden;">
      <div style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); padding: 1.5rem; color: white;">
        <h2 class="h5 mb-0 d-flex align-items-center gap-2" style="font-weight: 700;">
          <i class="bi bi-person-badge" style="font-size: 1.3rem;"></i>Your Profile
        </h2>
      </div>
      <div style="padding: 1.5rem;">
        <div style="display: flex; flex-direction: column; gap: 1.5rem;">
          <div style="padding: 1rem; background: #f9f9f9; border-radius: 8px;">
            <p style="margin: 0 0 0.5rem 0; color: #999; font-size: 0.9rem; font-weight: 600; text-transform: uppercase; letter-spacing: 0.05em;">
              <i class="bi bi-person-fill me-1" style="color: #f5576c;"></i>Name
            </p>
            <p style="margin: 0; font-weight: 700; color: #333; font-size: 1.1rem;">
              <?= htmlspecialchars($user['name']) ?>
            </p>
          </div>
          <div style="padding: 1rem; background: #f9f9f9; border-radius: 8px;">
            <p style="margin: 0 0 0.5rem 0; color: #999; font-size: 0.9rem; font-weight: 600; text-transform: uppercase; letter-spacing: 0.05em;">
              <i class="bi bi-envelope-fill me-1" style="color: #f5576c;"></i>Email
            </p>
            <p style="margin: 0; font-weight: 700; color: #333; font-size: 1.1rem;">
              <?= htmlspecialchars($user['email']) ?>
            </p>
          </div>
          <div style="padding: 1rem; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border-radius: 8px; color: white; text-align: center;">
            <p style="margin: 0; font-weight: 600; text-transform: uppercase; letter-spacing: 0.05em; opacity: 0.9; font-size: 0.9rem;">
              Role
            </p>
            <p style="margin: 0.5rem 0 0 0; font-weight: 700; font-size: 1.1rem;">
              <i class="bi bi-check-circle-fill me-1"></i>Student
            </p>
          </div>
        </div>
      </div>
    </section>

    <!-- Stats Card -->
    <section style="background: white; border-radius: 12px; padding: 0; box-shadow: 0 2px 8px rgba(0,0,0,0.08); overflow: hidden;">
      <div style="background: linear-gradient(135deg, #00d084 0%, #13c084 100%); padding: 1.5rem; color: white;">
        <h2 class="h5 mb-0 d-flex align-items-center gap-2" style="font-weight: 700;">
          <i class="bi bi-graph-up" style="font-size: 1.3rem;"></i>Progress
        </h2>
      </div>
      <div style="padding: 1.5rem;">
        <div style="display: flex; flex-direction: column; gap: 1.5rem;">
          <div style="padding: 1rem; background: #f0fdf4; border-left: 4px solid #00d084; border-radius: 8px;">
            <p style="margin: 0 0 0.5rem 0; color: #666; font-size: 0.9rem; font-weight: 600;">
              <i class="bi bi-check-circle-fill me-1" style="color: #00d084;"></i>Completed
            </p>
            <p style="margin: 0; font-weight: 700; color: #00d084; font-size: 1.8rem;">
              <?= count($completed) ?>
            </p>
          </div>
          <div style="padding: 1rem; background: #fff8e1; border-left: 4px solid #ffc107; border-radius: 8px;">
            <p style="margin: 0 0 0.5rem 0; color: #666; font-size: 0.9rem; font-weight: 600;">
              <i class="bi bi-play-circle-fill me-1" style="color: #ffc107;"></i>In Progress
            </p>
            <p style="margin: 0; font-weight: 700; color: #ffc107; font-size: 1.8rem;">
              <?= count($inprogress) ?>
            </p>
          </div>
          <div style="padding: 1rem; background: #e7f3ff; border-left: 4px solid #4facfe; border-radius: 8px;">
            <p style="margin: 0 0 0.5rem 0; color: #666; font-size: 0.9rem; font-weight: 600;">
              <i class="bi bi-clock-fill me-1" style="color: #4facfe;"></i>Unstarted
            </p>
            <p style="margin: 0; font-weight: 700; color: #4facfe; font-size: 1.8rem;">
              <?= count($unstarted) ?>
            </p>
          </div>
        </div>
      </div>
    </section>
  </div>
</div>

<script>
// Live countdown for in-progress remaining time on dashboard
(function(){
  function fmt(sec){ const m=Math.floor(sec/60), s=sec%60; return m+"m "+s.toString().padStart(2, '0')+"s"; }
  function tick(){
    document.querySelectorAll('.remain').forEach(function(el){
      let sec = parseInt(el.getAttribute('data-seconds')) || 0;
      if (sec <= 0) {
        el.textContent = 'Time Up';
        const item = el.closest('[data-attempt-id]');
        const btn = item ? item.querySelector('.resume-btn') : null;
        if (btn) { 
          btn.style.background = 'linear-gradient(135deg, #dc3545 0%, #c82333 100%)';
          btn.classList.add('disabled'); 
          btn.setAttribute('aria-disabled','true'); 
        }
        return;
      }
      sec = sec - 1;
      el.setAttribute('data-seconds', String(sec));
      el.textContent = fmt(sec);
      
      // Update card background based on remaining time
      const card = el.closest('[data-attempt-id]');
      if (card) {
        if (sec > 300) {
          card.style.background = '#e7f3ff';
          card.style.borderColor = '#4facfe';
        } else if (sec > 60) {
          card.style.background = '#fff8e1';
          card.style.borderColor = '#ffc107';
        } else {
          card.style.background = '#ffe7e7';
          card.style.borderColor = '#dc3545';
        }
      }
    });
    setTimeout(tick, 1000);
  }
  window.addEventListener('load', tick);
})();

// Poll every 5 seconds to refresh attempts and move from In Progress to Completed
(function(){
  function checkForCompletion(){
    const inProgressItems = document.querySelectorAll('[data-attempt-id]');
    if (inProgressItems.length === 0) return;
    
    const attemptIds = Array.from(inProgressItems).map(el => parseInt(el.getAttribute('data-attempt-id')));
    
    fetch('<?= htmlspecialchars(app_href('api/student_attempts_check.php')) ?>', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ attempt_ids: attemptIds })
    })
    .then(r => r.json())
    .then(data => {
      if (data.submitted_ids && data.submitted_ids.length > 0) {
        location.reload();
      }
    })
    .catch(err => console.error('Dashboard poll error:', err));
  }
  
  window.addEventListener('load', () => {
    setInterval(checkForCompletion, 5000);
  });
})();
</script>

<?php render_footer(); ?>

